import pandas as pd
import os
import numpy as np
from scipy.stats import entropy

def calcola_statistiche_descrittive(paziente_df):
    """
    Calcola media e statistiche descrittive di tutte le feature di un paziente.
    Restituisce due pandas Series:
    - media: solo media delle feature (con prefisso 'mean_')
    - stats: tutte le statistiche (mean, median, std, skew, kurtosis)
    """

    # Seleziona solo colonne numeriche
    paziente_df = paziente_df.select_dtypes(include='number')
    
    # Calcolo delle statistiche
    media = paziente_df.mean()
    mediana = paziente_df.median()
    skewness = paziente_df.skew()
    skewness = skewness.fillna(0)
    kurt = paziente_df.kurtosis()
    kurt = kurt.fillna(0) 
    q1 = paziente_df.quantile(0.25)
    q3 = paziente_df.quantile(0.75)
    iqr = q3 - q1
    
    # Entropia
    entropie = {}
    for col in paziente_df.columns:
        counts, _ = np.histogram(paziente_df[col], bins=10, density=True)
        if np.sum(counts) == 0:
            entropie[col] = 0    
        else:
            probs = counts / np.sum(counts)  
            entropie[col] = entropy(probs)

    entropia = pd.Series(entropie)
    entropia = entropia.fillna(0)
    
    # Etichette e aggregazione
    media.index = ["mean_" + col for col in paziente_df.columns]
    mediana.index = ["median_" + col for col in paziente_df.columns]
    skewness.index = ["skew_" + col for col in paziente_df.columns]
    kurt.index = ["kurt_" + col for col in paziente_df.columns]
    q1.index = ["25th_perc_" + col for col in paziente_df.columns]
    q3.index = ["75th_perc_" + col for col in paziente_df.columns]
    iqr.index = ["iqr_" + col for col in paziente_df.columns]
    entropia.index = ["entropy_" + col for col in paziente_df.columns]

    stats = pd.concat([media, mediana, skewness, kurt, entropia, iqr, q1, q3], axis=0)
    
    return media, stats

def aggregazione(lista_path, path_output):
    """
    Riceve una lista di path (uno per paziente) e salva:
    - un dataset con la media delle feature per paziente
    - un dataset con tutte le statistiche descrittive per paziente
    """
    
    medie_tutti = []
    stats_tutti = []
    dataset = [os.path.join(lista_path, f) for f in os.listdir(lista_path) if f.endswith('.csv')]
    
    for file_path in dataset:
        paziente_df = pd.read_csv(file_path)
        media, stats = calcola_statistiche_descrittive(paziente_df)
        medie_tutti.append(media)
        stats_tutti.append(stats)
    
    df_media = pd.DataFrame(medie_tutti)
    df_stats = pd.DataFrame(stats_tutti)
    
    # Aggiunge l'ID paziente e lo sposta all'inizio
    df_media["PatientID"] = [os.path.basename(f).split('_')[0] for f in dataset]
    df_stats["PatientID"] = [os.path.basename(f).split('_')[0] for f in dataset]
    
    df_media = df_media[["PatientID"] + [col for col in df_media.columns if col != "PatientID"]]
    df_stats = df_stats[["PatientID"] + [col for col in df_stats.columns if col != "PatientID"]]
    
    # Salvataggio
    df_media_path = os.path.join(path_output,"aggregazione_media.csv")
    df_stats_path = os.path.join(path_output,"aggregazione_stats.csv")
    df_media.to_csv(df_media_path, index=False)
    df_stats.to_csv(df_stats_path, index=False)
    df_media.to_excel(os.path.join(path_output, "aggregated_media.xlsx"), index=False)
    df_stats.to_excel(os.path.join(path_output, "aggregated_stats.xlsx"), index=False)
    
    return df_media_path, df_stats_path
